﻿///
/// アーカイブボリュームのフォーマット判定とコーデック取得を行う ArchiveDetector クラスのヘッダファイル。
///

#if       !defined(INCLUDED_SZ_ARCHIVE_DETECTOR)
#define            INCLUDED_SZ_ARCHIVE_DETECTOR

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

#if       !defined(INCLUDED_SZ_ARCHIVE_FORMAT)
#include  <szArchiveFormat.hpp>
#endif // !defined(INCLUDED_SZ_ARCHIVE_FORMAT)

#if       !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)
#include  <szForwardDeclarations.hpp>
#endif // !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)

SZ_NS_BEG(szpp)

//  【アーカイブ形式に対応したハンドラ】
//  7zip/CPP/7zip/Guid.txt に記載されているとおり、ZIP, RAR などの形式ごとのハンドラーには GUID が割り当てられている。
//    Handler GUIDs: {23170F69-40C1-278A-1000-000110xx0000}
//      01 Zip
//      02 BZip2
//      03 Rar
//      ...
//  LoadCodec.h で宣言されている CArcInfoEx::ClassID フィールドが、この GUID を保持しているので、
//  特定の形式に対応したハンドラーを見つけるには、まずハンドラーの GUID を調べ、CCodecs クラスが管理している
//  Formats メンバの各要素の ClassID フィールドと順次 memcmp していけばよい。

/// <summary>
/// アーカイブボリュームのフォーマット判定とコーデック取得を行うクラス。
/// </summary>
class SZ_SPEC ArchiveDetector
{
public:

//
// （主に）既存アーカイブに対するハンドラ作成
//

  /// <summary>
  /// 拡張子に対応する適切なハンドラがあればそのインデックスを返す。
  /// </summary>
  /// <remarks>
  /// あくまで拡張子による判定によると適切と思われるハンドラを返すだけであり、実際には不適切なハンドラが返されることもある。
  /// つまり、実際にアーカイブを開いて（列挙して）みてもエラーが起きなければ、そのときにはじめてハンドラが適切であったことになる。
  /// 特に、*.EXE のような拡張子ではマッチするハンドラが複数存在するし、そもそも拡張子による判定をすること自体が不適切である。
  /// </remarks>
  /// <param name="extension">判定する拡張子の文字列。先頭にピリオドが含まれていてはいけない。</param>
  /// <returns>ハンドラのポインタ。返されたポインタを delete するのは呼び出し側の責任となる。</returns>
  static ArchiveHandler *FindHandlerFromExtension(const szstring &extension);

  /// <summary>
  /// 入力ストリームに対応する適切なハンドラがあればそのインデックスを返す。
  /// </summary>
  /// <remarks>
  /// ストリーム内容の一部を調べて適切と思われるハンドラを返すだけであり、実際には不適切なハンドラが返されることもある。
  /// つまり、実際にアーカイブを開いて（列挙して）みてもエラーが起きなければ、そのときにはじめてハンドラが適切であったことになる。
  /// ただし、拡張子で判定する場合に比べるとかなり信頼性は高い。
  /// </remarks>
  /// <param name="stream">
  /// 判定するストリーム。メソッド呼び出しから戻った時点でストリームは解放されているので呼び出し前後で AddRef / Release を行う必要はない。
  /// IInStream の宣言については 7-Zip ソースツリーの <7zip\CPP\7zip\IStream.h> を参照。
  /// </param>
  /// <returns>ハンドラのポインタ。返されたポインタを delete するのは呼び出し側の責任となる。</returns>
  static ArchiveHandler *FindHandlerFromStream(IInStream *stream);

  /// <summary>
  /// ファイルに対応する適切なハンドラがあればそのインデックスを返す。
  /// </summary>
  /// <remarks>
  /// FindHandlerFromStream を利用して判定を行うので、信頼性は FindHandlerFromStream と同等。
  /// </remarks>
  /// <param name="fileName">判定するファイルのパス文字列。</param>
  /// <returns>ハンドラのポインタ。返されたポインタを delete するのは呼び出し側の責任となる。</returns>
  static ArchiveHandler *FindHandlerFromFile(const szstring &fileName);

//
// （主に）新規作成アーカイブに対するハンドラ作成
//

  /// <summary>
  /// アーカイブ形式を表す文字列に対応する適切なハンドラがあればそのインデックスを返す。
  /// </summary>
  /// <notes>
  /// 文字列は "ZIP", "Lzh" といったもので、7-Zip コーデックが持つ文字列に一致する必要があるため、現実的には本メソッドを使う場面があるとは思えない。
  /// 強いて言えば、7z.dll がバージョンアップされて新しい形式に対応したのに、SevenZip++ がそれをサポートしていないというような場合に、新しい形式のハンドラを作るために利用できる程度。
  /// </notes>
  /// <param name="name">ハンドラを作成したいアーカイブ形式の種類を表す文字列。大文字小文字は区別されない。</param>
  /// <returns>ハンドラのポインタ。返されたポインタを delete するのは呼び出し側の責任となる。</returns>
  static ArchiveHandler *FindHandlerFromName(const szstring &name);

  /// <summary>
  /// アーカイブ形式に対応する適切なハンドラがあればそのインデックスを返す。
  /// </summary>
  /// <param name="format">ハンドラを作成したいアーカイブ形式の種類を表す列挙値。</param>
  /// <returns>ハンドラのポインタ。返されたポインタを delete するのは呼び出し側の責任となる。</returns>
  static ArchiveHandler *FindHandlerFromFormat(ArchiveFormat format);

};

SZ_NS_END(szpp)

#endif // !defined(INCLUDED_SZ_ARCHIVE_DETECTOR)
